#ifndef __CVstEffect__
#define __CVstEffect__

#include <ECore.h>
#include <audioeffectx.h>
#include "CHostCanDo.hpp"
#include "CVstTempo.hpp"
#include "SVstProcessInformation.hpp"
#include "CVstMidiEventList.hpp"

using Exponent::Vst::CVstMidiEventList;
using Exponent::Vst::CHostCanDo;
using Exponent::Vst::CVstTempo;
using Exponent::Vst::SVstProcessInformation;

namespace Exponent
{
	namespace Vst
	{
		/**
		 * @class CVstEffect CVstEffect.hpp
		 * @brief Implements a Vst 2.3 AudioEffectX
		 *
		 * @date 06/08/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 * Please note that VST is copyright Steinberg Media GmBh. No challenge is made to any of their trademarks
		 * To use this file, you require a copy of the VST SDK, available from www.steinberg.net for free
		 *
		 * $Id: CVstEffect.hpp,v 1.6 2007/03/07 22:54:29 paul Exp $
		 */
		class CVstEffect : public AudioEffectX, public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */


//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construction
			 * @param audioMaster The VST audio master call back function
			 * @param numberOfParameters The number of parameters that the effect has...
			 * @param numberOfPresets The number of presets made available
			 */
			#ifdef VST_2_4_EXTENSIONS
			CVstEffect(audioMasterCallback audioMaster, const VstInt32 numberOfParameters, const VstInt32 numberOfPresets = 0);
			#else
			CVstEffect(audioMasterCallback audioMaster, const long numberOfParameters, const long numberOfPresets = 0);
			#endif

			/**
			 * Destruction
			 */
			virtual ~CVstEffect();

//	===========================================================================

			/**
			 * Resume processing
			 */
			virtual void resume();

			/**
			 * Process
			 * @param input The input audio buffers
			 * @param output The output audio buffer
			 * @param numberOfSamples The number of samples to process
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual void process(float **input, float **output, VstInt32 numberOfSamples);
			#else
			virtual void process(float **input, float **output, long numberOfSamples);
			#endif

			/**
			 * Process Replacing
			 * @param input The input audio buffers
			 * @param output The output audio buffer
			 * @param numberOfSamples The number of samples to process
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual void processReplacing(float **input, float **output, VstInt32 numberOfSamples);
			#else
			virtual void processReplacing(float **input, float **output, long numberOfSamples);
			#endif

			#ifdef VST_2_4_EXTENSIONS
			/**
			 * Process double replacing
			 * @param inputs The input audio buffers
			 * @param outputs The output audio buffer
			 * @param numberOfSamples The number of samples to process
			 */
			virtual void processDoubleReplacing(double **inputs, double **outputs, VstInt32 numberOfSamples);
			#endif

			/**
			 * Process VST midi events
			 * @param events the midi events
			 * @retval VstInt32 0 for more, undefined otherwise
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual VstInt32 processEvents(VstEvents *events);
			#else
			virtual long processEvents(VstEvents *events);
			#endif

//	===========================================================================

			/**
			 * Get the effect name
			 * @param name On return is filled with the name of the plugin
			 * @retval bool True if text filled properly, false otherwise
			 */
			virtual bool getEffectName(char* name);

			/**
			 * Get the vendor string
			 * @param text On return is filled with the name of the vendor
			 * @retval bool True if text filled properly, false otherwise
			 */
			virtual bool getVendorString(char* text);

			/**
			 * Get the product string
			 * @param text On return is filled with the name of the plugin
			 * @retval bool True if text filled properly, false otherwise
			 */
			virtual bool getProductString(char* text);

			/**
			 * Get host can do information
			 * @retval CHostCanDo The host can do
			 */
			CHostCanDo &getHostCanDo() { return m_hostCanDo; }

			/**
			 * Get the critical section
			 * @retval CCriticalSection* The critical section
			 */
			CCriticalSection *getCriticalSection() const { return m_criticalSection; }

			/**
			 * Get the midi channel
			 * @retval long The midi channel in the range 0 - 15
			 */
			long getMidiChannel() const { return m_midiEventList.getMidiChannel(); }

			#ifdef VST_2_4_EXTENSIONS
			/**
			 * Get the number of midi channels
			 * @retval VstInt32 Returns 1 currently, change if you have implemented more midi event lists
			 */
			VstInt32 getNumMidiInputChannels() { return 1; }
			#endif

//	===========================================================================

			/**
			 * Set the parameter
			 * @param index The index of the controls
			 * @param value The value to give the control
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual void setParameter(VstInt32 index, float value) { }
			#else
			virtual void setParameter(long index, float value) { }
			#endif

			/**
			 * Get the parameter
			 * @param index The index of the control
			 * @retval flot The value of the parameter
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual float getParameter(VstInt32 index) { return 0.f; }
			#else
			virtual float getParameter(long index) { return 0.f; }
			#endif

			/**
			 * Get the parameter label
			 * @param index The index of the controls
			 * @param label On return is filled with the label of the parameter
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual void getParameterLabel(VstInt32 index, char *label) { strcpy(label, "No param"); }
			#else
			virtual void getParameterLabel(long index, char *label) { strcpy(label, "No param"); }
			#endif

			/**
			 * Get the parameter display
			 * @param index The index of the controls
			 * @param text On return is filled with the string representing the parameter
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual void getParameterDisplay(VstInt32 index, char *text) { strcpy(text, "No param"); }
			#else
			virtual void getParameterDisplay(long index, char *text) { strcpy(text, "No param"); }
			#endif

			/**
			 * Get the parameter name
			 * @param index The index of the controls
			 * @param text On return is filled with the name of the parameter
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual void getParameterName(VstInt32 index, char *text) { strcpy(text, "No param"); }
			#else
			virtual void getParameterName(long index, char *text) { strcpy(text, "No param"); }
			#endif

			/**
			 * String to parameter
			 * @param index The index of the controls
			 * @param text The string to convert to a parameter
			 * @retval bool Always false, we dont suppor this
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual bool string2parameter(VstInt32 index, char *text) { return false; }
			#else
			virtual bool string2parameter(long index, char *text) { return false; }
			#endif

//	===========================================================================

			/**
			 * Set the block size
			 * @param blockSize The new blocksize
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual void setBlockSize(VstInt32 blockSize);
			#else
			virtual void setBlockSize(long blockSize);
			#endif

			/**
			 * Set the samplerate
			 * @param rate The new sample rate
			 */
			virtual void setSampleRate(float rate);

			/**
			 * Set the midi channel
			 * @param channel the midi channel in range 0 - 15
			 */
			virtual void setMidiChannel(const long channel);

			/**
			 * Do we support soft bypass
			 * @retval bool Always false, we dont support
			 */
			virtual bool setBypass(bool onOff) { return false; }

//	===========================================================================

			/**
			 * Set the program name
			 * @param name The name of the program
			 */
			virtual void setProgramName(char *name) { strcpy(name, "Default"); }

			/**
			 * Get the program name
			 * @param name On return is filled with the name of the current preset
			 */
			virtual void getProgramName(char *name) { strcpy(name, "Default"); }

			/**
			 * Set the program
			 * @param program The index of the program to load
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual void setProgram(VstInt32 program) { AudioEffectX::setProgram(program); }
			#else
			virtual void setProgram(long program) { AudioEffectX::setProgram(program); }
			#endif

			/**
			 * Get the program from an index
			 * @param category The catagory of the preset
			 * @param index The index of the preset
			 * @param text On return is filled with the preset name
			 * @retval bool Always false, we dont support this function
			 */
			#ifdef VST_2_4_EXTENSIONS
			virtual bool getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text) { strcpy(text, "Default"); return true; }
			#else
			virtual bool getProgramNameIndexed(long category, long index, char* text) { strcpy(text, "Default"); return true; }
			#endif

//	===========================================================================

		protected:

//	===========================================================================

			CHostCanDo m_hostCanDo;													/**< What can the host do? */
			CVstTempo m_vstTempo;													/**< Tempo and other time related information */
			SVstProcessInformation m_processInfo;									/**< The song setup information */
			CVstMidiEventList m_midiEventList;										/**< The midi event list */
			TPointerCollection< TPointerCollection<CLong> > m_midiControllers;		/**< The midi controllers */
			CCriticalSection *m_criticalSection;									/**< The ciritical section */
		};
	}
}

#endif		// End of CVstEffect.hpp